<?php
    // /classes/Shipment.php
    require_once __DIR__ . '/../config/functions.php';
    require_once __DIR__ . '/Utils.php';
    
    class Shipment {
        private $db;
    
        public function __construct() {
            global $db;
            $this->db = $db; 
        }
    
        // Generate a unique tracking number (prefix + random)
        private function generateTrackingNumber(): string {
            $tries = 0;
            
            do {
                $tracking = 'RT' . strtoupper(generateRandomString());
                $stmt = $this->db->prepare("SELECT id FROM shipments WHERE tracking_number = ?");
                $stmt->execute([$tracking]);
                $row = $stmt->fetch(PDO::FETCH_ASSOC);
                $tries++;
                if ($tries > 5) {
                    // fallback: append timestamp
                    $tracking .= time();
                }
            } while ($row);
            return $tracking;
        } 
    
        // Create shipment
        public function createShipment(array $data): array {
            global $db;
        
            // Generate tracking number
            $tracking_number = 'RT' . strtoupper(bin2hex(random_bytes(5)));
        
            $stmt = $db->prepare("
                INSERT INTO shipments (
                    tracking_number, sender_name, sender_address, 
                    recipient_name, recipient_address, recipient_email,
                    item_description, weight_kg, expected_delivery_date, 
                    notify_on_status_change, status, created_at, updated_at
                ) VALUES (
                    :tracking_number, :sender_name, :sender_address,
                    :recipient_name, :recipient_address, :recipient_email,
                    :item_description, :weight_kg, :expected_delivery_date,
                    :notify_on_status_change, 'PENDING', NOW(), NOW()
                )
            ");
        
            $stmt->execute([
                ':tracking_number' => $tracking_number,
                ':sender_name' => $data['sender_name'],
                ':sender_address' => $data['sender_address'],
                ':recipient_name' => $data['recipient_name'],
                ':recipient_address' => $data['recipient_address'],
                ':recipient_email' => $data['recipient_email'],
                ':item_description' => $data['contents'],
                ':weight_kg' => isset($data['weight_kg']) ? (float)$data['weight_kg'] : null,
                ':expected_delivery_date' => $data['expected_delivery_date'] ?? null,
                ':notify_on_status_change' => !empty($data['notify_on_status_change']) ? 1 : 0
            ]);
        
            return [
                'tracking_number' => $tracking_number,
                'sender_name' => $data['sender_name'],
                'recipient_name' => $data['recipient_name'],
                'status' => 'PENDING'
            ];
        }

        // Get shipment by id
        public function getShipmentById(int $id): ?array {
            $stmt = $this->db->prepare("SELECT * FROM shipments WHERE id = ?");
            $stmt->execute([$id]);
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            return $row ?: null;
        }
    
        // Get shipment by tracking number
        public function getByTracking(string $trackingNumber): ?array {
            $stmt = $this->db->prepare("SELECT * FROM shipments WHERE tracking_number = ?");
            $stmt->execute([$trackingNumber]);
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            if (!$row) return null;
        
            // normalize for FE
            $row['id'] = (string)$row['id'];
            $row['status'] = strtolower($row['status']);
            $row['recipient_email'] = $row['recipient_email'] ?? null;
            $row['notify_on_status_change'] = (bool)($row['notify_on_status_change'] ?? false);
            $row['last_email_sent_at'] = $row['last_email_sent_at'] ?? null;
            $row['last_email_status'] = $row['last_email_status'] ?? null;
        
            return $row;
        }
    
        // Add shipment event
        public function addEvent(int $shipmentId, string $eventType, string $message): array {
            // ensure shipment exists
            $shipment = $this->getShipmentById($shipmentId);
            if (!$shipment) throw new Exception("Shipment not found");
    
            $stmt = $this->db->prepare("INSERT INTO shipment_events (shipment_id, event_type, event_message, created_at) VALUES (?, ?, ?, ?)");
            $now = date('Y-m-d H:i:s');
            $stmt->execute([$shipmentId, $eventType, $message, $now]);
    
            // Optionally return the event that was created
            $eventId = $this->db->lastInsertId();
            $stmt2 = $this->db->prepare("SELECT * FROM shipment_events WHERE id = ?");
            $stmt2->execute([$eventId]);
            return $stmt2->fetch(PDO::FETCH_ASSOC);
        }
    
        // Update shipment status (and append status history as event)
        public function updateStatus(int $shipmentId, string $newStatus, string $note = ""): array {
            $valid = ['PENDING','IN_TRANSIT','PICKED_UP','OUT_FOR_DELIVERY','DELIVERED'];
            if (!in_array($newStatus, $valid)) {
                throw new Exception("Invalid status");
            }
            $shipment = $this->getShipmentById($shipmentId);
            if (!$shipment) throw new Exception("Shipment not found");
    
            $stmt = $this->db->prepare("UPDATE shipments SET status = ?, updated_at = ? WHERE id = ?");
            $now = date('Y-m-d H:i:s');
            $stmt->execute([$newStatus, $now, $shipmentId]);
    
            // add an event describing the status change
            $message = $note !== "" ? $note : "Status changed to {$newStatus}";
            $this->addEvent($shipmentId, "status_change", $message);
    
            return $this->getShipmentById($shipmentId);
        }
    
        // List shipments (admin) with optional filters
        public function listShipments(array $filters = [], int $limit = 50, int $offset = 0): array {
            $sql = "SELECT * FROM shipments WHERE 1=1";
            $params = [];
            if (!empty($filters['status'])) {
                $sql .= " AND status = ?";
                $params[] = $filters['status'];
            }
            if (!empty($filters['tracking'])) {
                $sql .= " AND tracking_number = ?";
                $params[] = $filters['tracking'];
            }
            $sql .= " ORDER BY created_at DESC LIMIT ? OFFSET ?";
            $stmt = $this->db->prepare($sql);
            $i = 1;
            foreach ($params as $p) {
                $stmt->bindValue($i, $p);
                $i++;
            }
            $stmt->bindValue($i++, $limit, PDO::PARAM_INT);
            $stmt->bindValue($i, $offset, PDO::PARAM_INT);
            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        }
        
        public function addEventFromAdmin(int $shipmentId, string $status, ?string $location, ?string $description): array 
        {
            // Check shipment exists
            $shipment = $this->getShipmentById($shipmentId);
            if (!$shipment) {
                throw new Exception("Shipment not found");
            }
        
            // Insert event
            $stmt = $this->db->prepare("
                INSERT INTO shipment_events (shipment_id, event_type, location, event_message, created_at)
                VALUES (?, ?, ?, ?, ?)
            ");
            
            $now = date('Y-m-d H:i:s');
            $stmt->execute([$shipmentId, $status, $location, $description, $now]);
            $eventId = $this->db->lastInsertId();
        
            // Update shipment status in main table
            $stmt2 = $this->db->prepare("
                UPDATE shipments SET status = ?, updated_at = ? WHERE id = ?
            ");
            $stmt2->execute([$status, $now, $shipmentId]);
        
            // Return created event
            $stmt3 = $this->db->prepare("SELECT * FROM shipment_events WHERE id = ?");
            $stmt3->execute([$eventId]);
        
            return $stmt3->fetch(PDO::FETCH_ASSOC) ?? [];
        }

    
        // get events for a shipment
        public function getEvents(int $shipmentId): array {
            $stmt = $this->db->prepare("
                SELECT id, LOWER(event_type) AS status, event_message AS description, location, created_at
                FROM shipment_events
                WHERE shipment_id = ?
                ORDER BY created_at ASC
            ");
            $stmt->execute([$shipmentId]);
            $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
            // cast ids to string
            foreach ($rows as &$row) {
                $row['id'] = (string)$row['id'];
                $row['location'] = $row['location'] ?? null;
            }
            return $rows;
        }

    }
