<?php
    require_once __DIR__ . '/../config/functions.php';

    class Booking {
        private $db;

        public function __construct() {
            global $db;
            $this->db = $db;
        }

        public function createBooking(int $tripId, string $passengerName, string $passengerEmail, int $seats, float $amount): array {
            $reference = generateRandomString(10);
            $stmt = $this->db->prepare("INSERT INTO bookings (trip_id, passenger_name, passenger_email, seats, amount, reference, created_at) VALUES (?, ?, ?, ?, ?, ?, ?)");
            $now = date('Y-m-d H:i:s');
            $stmt->execute([$tripId, $passengerName, $passengerEmail, $seats, $amount, $reference, $now]);
            $bookingId = $this->db->lastInsertId();
            return $this->get($bookingId);
        }

        public function get(int $id): array {
            $stmt = $this->db->prepare("SELECT * FROM bookings WHERE id = ?");
            $stmt->execute([$id]);
            return $stmt->fetch(PDO::FETCH_ASSOC);
        }

        public function list(int $tripId = null, int $limit = 50, int $offset = 0, $status = null): array {
            $sql = "SELECT * FROM bookings";
            $params = [];
            if ($tripId !== null) {
                $sql .= " WHERE trip_id = ?";
                $params[] = $tripId;
            }
            $sql .= " ORDER BY created_at DESC LIMIT ? OFFSET ?";
            $params[] = $limit;
            $params[] = $offset;

            $stmt = $this->db->prepare($sql);
            $stmt->execute($params);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        }

        public function count() {
            global $db;
            $stmt = $db->query("SELECT COUNT(*) as total FROM bookings");
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            return (int)$row['total'];
        }

        public function today() {
            global $db;
            $today = date('Y-m-d');
            $stmt = $db->prepare("SELECT * FROM bookings WHERE DATE(created_at) = ?");
            $stmt->execute([$today]);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        }

        public function getByTrackingNumber($trackingNumber) {
            global $db;
            $stmt = $db->prepare("SELECT * FROM bookings WHERE reference  = :tracking");
            $stmt->execute(['tracking' => $trackingNumber]);
            return $stmt->fetch(PDO::FETCH_ASSOC);
        }

        public function getByReference(string $reference): ?array {
            $stmt = $this->db->prepare("SELECT * FROM bookings WHERE reference = ?");
            $stmt->execute([$reference]);
            $booking = $stmt->fetch(PDO::FETCH_ASSOC);
            return $booking ?: null;
        }

        public function verifyPayment(int $bookingId, string $paymentReference): array {
            $stmt = $this->db->prepare("SELECT * FROM bookings WHERE id = ?");
            $stmt->execute([$bookingId]);
            $booking = $stmt->fetch(PDO::FETCH_ASSOC);
            if (!$booking) throw new Exception("Booking not found");
        
            // Update status to 'paid' and store reference
            $stmt = $this->db->prepare("UPDATE bookings SET status = 'paid', trx_refrence= ? WHERE id = ?");
            $stmt->execute([$paymentReference, $bookingId]);
        
            return $this->get($bookingId); 
        }

    }
