<?php
    require_once __DIR__ . '/../config/functions.php';
    require_once __DIR__ . '/AdminUser.php';
    require_once __DIR__ . '/Utils.php';

    class Auth {
        public function __construct() {
            global $db;
            $this->db = $db;
            $this->adminUser = new AdminUser();
        }
        

        public function generateToken($user) 
        {
            $payload = [
                'id' => $user['id'] ?? $user->id,
                'role' => $user['role'] ?? $user->role, // include role for auth checks
                'ts' => time()
            ];
        
            $base = base64_encode(json_encode($payload));
            $signature = hash_hmac('sha256', $base, AUTH_SECRET_KEY);
        
            return $base . "." . $signature;
        }

        
    
        // Verify token
        public function verifyToken($token)
        {
            if (!$token || !str_contains($token, ".")) {
                return false;
            }
    
            list($base, $signature) = explode(".", $token);
    
            $expected = hash_hmac('sha256', $base, AUTH_SECRET_KEY);
    
            if (!hash_equals($expected, $signature)) {
                return false;
            }
    
            return json_decode(base64_decode($base), true);
        }
        
        
        public function login(string $email, string $password): array {
            $stmt = $this->db->prepare("SELECT id, password, full_name, role, active FROM users WHERE email = ?");
            $stmt->execute([$email]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$user || !$user['is_active']) {
                throw new Exception("Invalid credentials");
            }

            if (!password_verify($password, $user['password'])) {
                throw new Exception("Invalid credentials");
            }

            $token = $this->generateJWT($user['id'], $user['role']);

            return [
                'access_token' => $token,
                'user' => [
                    'id' => $user['id'],
                    'email' => $email,
                    'full_name' => $user['full_name'],
                    'role' => $user['role']
                ]
            ];
        }


        public function register(string $email, string $password, string $fullName): array {
            return $this->adminUser->createAdmin($email, $password, $fullName, 'admin');
        }


        public function changePassword(string $userId, string $oldPassword, string $newPassword): bool {
            $stmt = $this->db->prepare("SELECT password FROM users WHERE id = ?");
            $stmt->execute([$userId]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$user) {
                throw new Exception("User not found");
            }

            if (!password_verify($oldPassword, $user['password'])) {
                throw new Exception("Old password does not match");
            }

            $hash = password_hash($newPassword, PASSWORD_BCRYPT);
            $stmt = $this->db->prepare("UPDATE users SET password = ?, updated_at = ? WHERE id = ?");
            $stmt->execute([$hash, date('Y-m-d H:i:s'), $userId]);
            return true;
        }
    }
