<?php
    require_once __DIR__ . '/../config/functions.php';

    class AdminUser {
        private $db;

        public function __construct() {
            global $db;
            $this->db = $db;
        }

        public function createAdmin(string $email, string $password, string $fullName, string $role): array {
            $hash = password_hash($password, PASSWORD_BCRYPT);
            $stmt = $this->db->prepare("INSERT INTO admin_users (email, password, full_name, role, active, created_at, updated_at) VALUES (?, ?, ?, ?, 1, ?, ?)");
            $now = date('Y-m-d H:i:s');
            $stmt->execute([$email, $hash, $fullName, $role, $now, $now]);
            $id = $this->db->lastInsertId();
            return $this->get($id);
        }

        public function get($id): array {
            $stmt = $this->db->prepare("SELECT id, email, full_name, role, active FROM admin_users WHERE id = ?");
            $stmt->execute([$id]);
            return $stmt->fetch(PDO::FETCH_ASSOC);
        }
        
        public function login(string $email, string $password): array {
            $stmt = $this->db->prepare("SELECT * FROM admin_users WHERE email = :email LIMIT 1");
            $stmt->execute(['email' => $email]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
            if (!$user) {
                throw new Exception('Invalid credentials');
            }
    
            // Verify password
            if (!password_verify($password, $user['password'])) {
                throw new Exception('Invalid credentials');
            }
    
            unset($user['password']);
    
            return $user;
        }

        public function list(int $limit = 50, int $offset = 0): array {
            $stmt = $this->db->prepare("SELECT id, email, full_name, role, active FROM admin_users ORDER BY id DESC LIMIT ? OFFSET ?");
            $stmt->bindValue(1, $limit, PDO::PARAM_INT);
            $stmt->bindValue(2, $offset, PDO::PARAM_INT);
            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        }

        public function updateRole(int $id, string $role): bool {
            $stmt = $this->db->prepare("UPDATE admin_users SET role = ?, updated_at = ? WHERE id = ?");
            return $stmt->execute([$role, date('Y-m-d H:i:s'), $id]);
        }

        public function setPassword(int $id, string $password): bool {
            $hash = password_hash($password, PASSWORD_BCRYPT);
            $stmt = $this->db->prepare("UPDATE admin_users SET password = ?, updated_at = ? WHERE id = ?");
            return $stmt->execute([$hash, date('Y-m-d H:i:s'), $id]);
        }

        
        public function verifyPassword(int $id, string $oldPassword): bool {
            $stmt = $this->db->prepare("SELECT password FROM admin_users WHERE id = ?");
            $stmt->execute([$id]);
            $user = $stmt->fetch();
        
            if (!$user) return false;
        
            return password_verify($oldPassword, $user['password']);
        }

    }
